#Replication package for BPEA Paper:
#The Emergence of a Uniform Business Cycle in the United States: Evidence from New Claims-Based Unemployment Data

#Andrew Fieldhouse, David Munro, Christoffer Koch, and Sean Howard

#Generates Figures 3, 4, 9, 10

script_dir <- dirname(rstudioapi::getActiveDocumentContext()$path)
setwd(script_dir)
getwd()
setwd('..')

library(tidyverse)
library(haven)

main_df <- read_dta("Rep_data/Fitted CBURs.dta")

main_df$Date <- as.character(main_df$Date)
main_df$DateR<-as.Date(main_df$Date,format = "%Y-%m-%d")

main_df <- main_df %>% 
  group_by(Date) %>%
  mutate(SD_RelUR=sd(D_ClaimsUR,na.rm=T))

main_df <- main_df %>%  
  group_by(State) %>%
  mutate(EmpGrowthAnnual = (log(nonfarm_SA)-lag(log(nonfarm_SA),n=12))*100,
         EmpGrowthAnnualUS = (log(US_NFE_SA)-lag(log(US_NFE_SA),n=12))*100,
         RelEmpGrowthUS=EmpGrowthAnnual-EmpGrowthAnnualUS,
         EmpGrowth10years=lag(EmpGrowthAnnual,n=120))

main_df <- main_df %>% 
  group_by(Date) %>%
  mutate(SD_RelEmp=sd(RelEmpGrowthUS,na.rm=T))

recessions.df = read.table(textConnection(
  "Peak, Trough
1948-11-01, 1949-10-01
1953-07-01, 1954-05-01
1957-08-01, 1958-04-01
1960-04-01, 1961-02-01
1969-12-01, 1970-11-01
1973-11-01, 1975-03-01
1980-01-01, 1980-07-01
1981-07-01, 1982-11-01
1990-07-01, 1991-03-01
2001-03-01, 2001-11-01
2007-12-01, 2009-06-01
2020-03-01, 2020-04-01"), sep=',',
  colClasses=c('Date', 'Date'), header=TRUE)

#Figure 3b: Relative CBURs and Dispersion
Fig3b<-ggplot() + 
  geom_line(data = main_df, aes(x = DateR, y = D_ClaimsUR, group = State), color = "black", linewidth = 0.8, alpha = 0.2) + 
  theme(legend.position = "none") +
  geom_rect(data = recessions.df, aes(xmin = Peak, xmax = Trough, ymin = -Inf, ymax = +Inf), fill = 'grey', alpha = 0.4) +
  xlab("") + ylab("Relative CBUR") +
  geom_line(data = filter(main_df, State == "AK"), aes(x = DateR, y = SD_RelUR*4), color = "black", linewidth = 0.8, linetype = "dotdash") + 
  theme(legend.position = "none") +
  theme_bw() +
  scale_y_continuous(sec.axis = sec_axis(trans=~./4, name = "Std. Dev. Relative CBUR"))

ggsave("Rep_Figures/Fig3b_Relative_CBUR.pdf", plot = Fig3b, width = 6, height = 4)

#Figure 3a: Relative Emp Growth and Dispersion
Fig3a<-ggplot() + 
  geom_line(data = main_df, aes(x = DateR, y = RelEmpGrowthUS, group = State), color = "black", size = 0.8, alpha = 0.2) + 
  theme(legend.position = "none") +
  geom_rect(data = recessions.df, aes(xmin = Peak, xmax = Trough, ymin = -Inf, ymax = +Inf), fill = 'grey', alpha = 0.4) +
  xlab("") + ylab("Relative Employment Growth Rate") +
  geom_line(data = filter(main_df, State == "AK"), aes(x = DateR, y = SD_RelEmp*5), color = "black", linewidth = 0.8, linetype = "dotdash") + 
  theme(legend.position = "none") +
  theme_bw() +
  scale_y_continuous(sec.axis = sec_axis(trans=~./5, name = "Std. Dev. Relative Emp. Growth"))

ggsave("Rep_Figures/Fig3a_Relative_EmpGrowth.pdf", plot = Fig3a, width = 6, height = 4)

Fig3_df<-main_df[,c('DateR', 'State', 'RelEmpGrowthUS', 'SD_RelEmp','D_ClaimsUR','SD_RelUR')]
library(openxlsx)
write.xlsx(Fig3_df, 'Rep_Figures_Data_Export/Fig3_Export.xlsx')

#Constructing Figure 4
main_df_50s<- main_df %>%
  filter(Date < as.Date("1960-01-01"))%>%
  group_by(State)%>%
  summarise(UR1950sRel=mean(D_ClaimsUR, na.rm=T),
            EmpGrowth1950s=mean(EmpGrowthAnnual, na.rm=T))

main_df_60s<- main_df %>%
  filter(Date >= as.Date("1960-01-01")&Date < as.Date("1970-01-01"))%>%
  group_by(State)%>%
  summarise(UR1960sRel=mean(D_ClaimsUR, na.rm=T),
            EmpGrowth1960s=mean(EmpGrowthAnnual, na.rm=T))

main_df_70s<- main_df %>%
  filter(Date >= as.Date("1970-01-01")&Date < as.Date("1980-01-01"))%>%
  group_by(State)%>%
  summarise(UR1970sRel=mean(D_ClaimsUR, na.rm=T),
            EmpGrowth1970s=mean(EmpGrowthAnnual, na.rm=T))

main_df_80s<- main_df %>%
  filter(Date >= as.Date("1980-01-01")&Date < as.Date("1990-01-01"))%>%
  group_by(State)%>%
  summarise(UR1980sRel=mean(D_ClaimsUR, na.rm=T),
            EmpGrowth1980s=mean(EmpGrowthAnnual, na.rm=T))

main_df_90s<- main_df %>%
  filter(Date >= as.Date("1990-01-01")&Date < as.Date("2000-01-01"))%>%
  group_by(State)%>%
  summarise(UR1990sRel=mean(D_ClaimsUR, na.rm=T),
            EmpGrowth1990s=mean(EmpGrowthAnnual, na.rm=T))

main_df_2000s<- main_df %>%
  filter(Date >= as.Date("2000-01-01")&Date < as.Date("2010-01-01"))%>%
  group_by(State)%>%
  summarise(UR2000sRel=mean(D_ClaimsUR, na.rm=T),
            EmpGrowth2000s=mean(EmpGrowthAnnual, na.rm=T))

main_df_2010s<- main_df %>%
  filter(Date >= as.Date("2010-01-01")&Date < as.Date("2020-01-01"))%>%
  group_by(State)%>%
  summarise(UR2010sRel=mean(D_ClaimsUR, na.rm=T),
            EmpGrowth2010s=mean(EmpGrowthAnnual, na.rm=T))

df_list_UR<-list(main_df_50s, main_df_60s, main_df_70s, main_df_80s, main_df_90s, main_df_2000s,main_df_2010s)
main_df_long_Decades<-Reduce(function(x, y) merge(x, y, all=TRUE), df_list_UR)

#UR plots with decade pooling
#1950s
UR_Plot_50s<-ggplot(main_df_long_Decades, aes(x=UR1950sRel, y=UR1960sRel))+
  geom_point(alpha=0.2)+ geom_smooth(method=lm, se=FALSE, color="black")+ylim(-4,4)+xlim(-4,4)+
  ylab("Relative CBUR 1960s")+xlab("Relative CBUR 1950s")+ theme_bw()

model_UR_50s<-lm(UR1960sRel~UR1950sRel,data=main_df_long_Decades)
summary(model_UR_50s)

#1960s
UR_Plot_60s<-ggplot(main_df_long_Decades, aes(x=UR1960sRel, y=UR1970sRel))+
  geom_point(alpha=0.2)+ geom_smooth(method=lm, se=FALSE, color="black")+ylim(-4,4)+xlim(-4,4)+
  ylab("Relative CBUR 1970s")+xlab("Relative CBUR 1960s")+ theme_bw()

model_UR_60s<-lm(UR1970sRel~UR1960sRel,data=main_df_long_Decades)
summary(model_UR_60s)

#1970s
UR_Plot_70s<-ggplot(main_df_long_Decades, aes(x=UR1970sRel, y=UR1980sRel))+
  geom_point(alpha=0.2)+ geom_smooth(method=lm, se=FALSE, color="black")+ylim(-4, 4)+xlim(-4,4)+
  ylab("Relative CBUR 1980s")+xlab("Relative CBUR 1970s")+theme_bw()

model_UR_70s<-lm(UR1980sRel~UR1970sRel,data=main_df_long_Decades)
summary(model_UR_70s)

#1980s
UR_Plot_80s<-ggplot(main_df_long_Decades, aes(x=UR1980sRel, y=UR1990sRel))+
  geom_point(alpha=0.2)+ geom_smooth(method=lm, se=FALSE, color="black")+ylim(-4,4)+xlim(-4,4)+
  ylab("Relative CBUR 1990s")+xlab("Relative CBUR 1980s")+ theme_bw()

model_UR_80s<-lm(UR1990sRel~UR1980sRel,data=main_df_long_Decades)
summary(model_UR_80s)

#1990s
UR_Plot_90s<-ggplot(main_df_long_Decades, aes(x=UR1990sRel, y=UR2000sRel))+
  geom_point(alpha=0.2)+ geom_smooth(method=lm, se=FALSE, color="black")+ylim(-4,4)+xlim(-4,4)+
  ylab("Relative CBUR 2000s")+xlab("Relative CBUR 1990s")+ theme_bw()

model_UR_90s<-lm(UR2000sRel~UR1990sRel,data=main_df_long_Decades)
summary(model_UR_90s)

#2000s
UR_Plot_2000s<-ggplot(main_df_long_Decades, aes(x=UR2000sRel, y=UR2010sRel))+
  geom_point(alpha=0.2)+ geom_smooth(method=lm, se=FALSE, color="black")+ylim(-4,4)+xlim(-4,4)+
  ylab("Relative CBUR 2010s")+xlab("Relative CBUR 2000s")+ theme_bw()

model_UR_2000s<-lm(UR2010sRel~UR2000sRel,data=main_df_long_Decades)
summary(model_UR_2000s)

library(ggpubr)
CombinedFigUR <- ggarrange(UR_Plot_50s, UR_Plot_60s, UR_Plot_70s,UR_Plot_80s,UR_Plot_90s,UR_Plot_2000s,
                           ncol = 2, nrow = 3)
#Figure 4b
CombinedFigUR

ggsave("Rep_Figures/Fig4b_CBUR Decades.pdf", plot = CombinedFigUR, width = 8, height = 6)

#employment

#1950s
Emp_Plot_50s<-ggplot(main_df_long_Decades, aes(x=EmpGrowth1950s, y=EmpGrowth1960s))+
  geom_point(alpha=0.2)+ geom_smooth(method=lm, se=FALSE, color="black")+ylim(-2,8)+xlim(-2,8)+
  ylab("Emp Growth 1960s")+xlab("Emp Growth 1950s")+ theme_bw()

model_50s<-lm(EmpGrowth1960s~EmpGrowth1950s,data=main_df_long_Decades)
summary(model_50s)

#1960s
Emp_Plot_60s<-ggplot(main_df_long_Decades, aes(x=EmpGrowth1960s, y=EmpGrowth1970s))+
  geom_point(alpha=0.2)+ geom_smooth(method=lm, se=FALSE, color="black")+ylim(-2,8)+xlim(-2,8)+
  ylab("Emp Growth 1970s")+xlab("Emp Growth 1960s")+ theme_bw()

model_60s<-lm(EmpGrowth1970s~EmpGrowth1960s,data=main_df_long_Decades)
summary(model_60s)

#1970s
Emp_Plot_70s<-ggplot(main_df_long_Decades, aes(x=EmpGrowth1970s, y=EmpGrowth1980s))+
  geom_point(alpha=0.2)+ geom_smooth(method=lm, se=FALSE, color="black")+ylim(-2,8)+xlim(-2,8)+
  ylab("Emp Growth 1980s")+xlab("Emp Growth 1970s")+ theme_bw()

model_70s<-lm(EmpGrowth1980s~EmpGrowth1970s,data=main_df_long_Decades)
summary(model_70s)

#1980s
Emp_Plot_80s<-ggplot(main_df_long_Decades, aes(x=EmpGrowth1980s, y=EmpGrowth1990s))+
  geom_point(alpha=0.2)+ geom_smooth(method=lm, se=FALSE, color="black")+ylim(-2,8)+xlim(-2,8)+
  ylab("Emp Growth 1990s")+xlab("Emp Growth 1980s")+ theme_bw()

model_80s<-lm(EmpGrowth1990s~EmpGrowth1980s,data=main_df_long_Decades)
summary(model_80s)

#1990s
Emp_Plot_90s<-ggplot(main_df_long_Decades, aes(x=EmpGrowth1990s, y=EmpGrowth2000s))+
  geom_point(alpha=0.2)+ geom_smooth(method=lm, se=FALSE, color="black")+ylim(-2,8)+xlim(-2,8)+
  ylab("Emp Growth 2000s")+xlab("Emp Growth 1990s")+ theme_bw()

model_90s<-lm(EmpGrowth2000s~EmpGrowth1990s,data=main_df_long_Decades)
summary(model_90s)

#2000s
Emp_Plot_2000s<-ggplot(main_df_long_Decades, aes(x=EmpGrowth2000s, y=EmpGrowth2010s))+
  geom_point(alpha=0.2)+ geom_smooth(method=lm, se=FALSE, color="black")+ylim(-2,8)+xlim(-2,8)+
  ylab("Emp Growth 2010s")+xlab("Emp Growth 2000s")+ theme_bw()

model_2000s<-lm(EmpGrowth2010s~EmpGrowth2000s,data=main_df_long_Decades)
summary(model_2000s)

CombinedFigEmp <- ggarrange(Emp_Plot_50s, Emp_Plot_60s, Emp_Plot_70s,Emp_Plot_80s,Emp_Plot_90s,Emp_Plot_2000s,
                            ncol = 2, nrow = 3)
#Figure 4a
CombinedFigEmp

ggsave("Rep_Figures/Fig4a_Emp Growth Decades.pdf", plot = CombinedFigEmp, width = 8, height = 6)

write.xlsx(main_df_long_Decades, 'Rep_Figures_Data_Export/Fig4_Export.xlsx')

#Construct the amplitude figures

library("readxl")
UR_CB_Pred <- read_excel("Rep_Data/Dataset_UR_Pred.xlsx")

colnames(UR_CB_Pred) <- gsub("UR_Pred", "", colnames(UR_CB_Pred))
for (i in 1:length(names(UR_CB_Pred))) {
  if (!grepl("DateR", names(UR_CB_Pred)[i])) {
    names(UR_CB_Pred)[i] <- paste("URPred_", names(UR_CB_Pred)[i], sep = "")
  }
}

StateDatesLongPred <- read_excel("Rep_Data/StateRecessionDates.xlsx")

merged_dataset_Pred <- merge(StateDatesLongPred, UR_CB_Pred, by = "DateR")

df_long_Pred <- pivot_longer(merged_dataset_Pred, cols = -DateR, names_to = "Column", values_to = "Value")

df_long_Pred <- df_long_Pred %>%
  separate(Column, c("Recession_Ptr", "State_Abbreviation"), sep = "_")

df_long_Pred <- pivot_wider(
  df_long_Pred,
  id_cols = c("DateR", "State_Abbreviation"),
  names_from = Recession_Ptr,
  values_from = Value
)

df_long_Pred <- df_long_Pred %>%
  arrange(State_Abbreviation, DateR)

df_long_Pred <- df_long_Pred %>%
  filter(Peak == 1 | Trough == 1)

df_long_trimmed_Pred <- df_long_Pred %>%
  select(-Recession)

df_long_trimmed_Pred <- df_long_trimmed_Pred %>%
  arrange(State_Abbreviation, DateR) %>%
  group_by(State_Abbreviation) %>%
  mutate(Amplitude = URPred - lag(URPred)) %>%
  ungroup()

df_long_trimmed_Pred <- df_long_trimmed_Pred %>%
  mutate(Expansion = ifelse(Peak == 1, Amplitude, NA),
         Contraction = ifelse(Trough == 1, Amplitude, NA))

df_long_trimmed_Pred <- df_long_trimmed_Pred %>%
  mutate(Abs_Expansion = abs(Expansion),
         Abs_Contraction = abs(Contraction))

df_long_trimmed_Pred$Next_Abs_Expansion <- lead(df_long_trimmed_Pred$Abs_Expansion)
df_long_trimmed_Pred$Time_to_Next_Peak_Days <- ifelse(is.na(df_long_trimmed_Pred$Next_Abs_Expansion), NA, lead(df_long_trimmed_Pred$DateR) - df_long_trimmed_Pred$DateR)
df_long_trimmed_Pred$Time_to_Next_Peak_Days <- as.integer(df_long_trimmed_Pred$Time_to_Next_Peak_Days)


df_long_trimmed_Pred<-df_long_trimmed_Pred %>% drop_na(DateR)
df_long_trimmed_Pred$Next_Abs_Contraction <- lead(df_long_trimmed_Pred$Abs_Contraction)
df_long_trimmed_Pred$Time_to_Next_Trough_Days <- ifelse(is.na(df_long_trimmed_Pred$Next_Abs_Contraction), NA, lead(df_long_trimmed_Pred$DateR) - df_long_trimmed_Pred$DateR)
df_long_trimmed_Pred$Time_to_Next_Trough_Days <- as.integer(df_long_trimmed_Pred$Time_to_Next_Trough_Days)

df_long_trimmed_Pred$Color <- ifelse(df_long_trimmed_Pred$DateR < as.Date("1980-01-01"), "Before 1980", "After 1980")

# Filter out pandemic
df_nocovid_Pred <- df_long_trimmed_Pred[df_long_trimmed_Pred$DateR < as.Date("2020-01-01"), ]

#Figure 10: Right Figure 
DNS_Right<-ggplot(filter(df_nocovid_Pred,Abs_Expansion>0), aes(x = Abs_Expansion, y = Next_Abs_Contraction,color=Color)) +
  geom_point(aes(shape=Color), alpha = 0.5)+scale_shape_manual(values=c(16, 3))+geom_smooth(method = "lm", aes(group = Color, linetype=Color), se=FALSE, size=1.2)+theme(axis.line = element_line(colour = "black"),panel.background = element_blank(), legend.position="top",legend.title=element_blank())+
  scale_color_manual(values=c('grey50','black'))+xlab("Amplitude of a expansion")+ylab("Amplitude of the subsequent contraction")+ guides(factor = "none")+xlim(0, 15)+ylim(0, 15)+
  geom_abline(intercept = 0, slope = 1, size = 0.5,linetype = "dotdash")

ggsave("Rep_Figures/Fig10b_DNS_CBUR_Fit_Right.pdf", plot = DNS_Right, width = 6, height = 4)

model1 <- lm(Next_Abs_Contraction ~ Abs_Expansion, data=filter(df_nocovid_Pred,Color=="Before 1980"))
summary(model1)
model1a <- lm(Next_Abs_Contraction ~ Abs_Expansion, data=filter(df_nocovid_Pred,Color=="After 1980"))
summary(model1a)

#Figure 10: Left Figure 
DNS_Left<-ggplot(filter(df_nocovid_Pred,Abs_Contraction>0), aes(x=Abs_Contraction, y = Next_Abs_Expansion,color=Color)) +
  geom_point(aes(shape=Color),alpha = 0.5)+scale_shape_manual(values=c(16, 3))+geom_smooth(method = "lm", aes(group = Color, linetype=Color), se=FALSE, size=1.2)+theme(axis.line = element_line(colour = "black"),panel.background = element_blank(), legend.position="top",legend.title=element_blank())+
  scale_color_manual(values=c('grey50','black')) + xlab("Amplitude of a contraction")+ylab("Amplitude of the subsequent expansion")+ guides(factor = "none")+xlim(0, 15)+ylim(0, 15)+
  geom_abline(intercept = 0, slope = 1, size = 0.5,,linetype = "dotdash")

ggsave("Rep_Figures/Fig10a_DNS_CBUR_Fit_Left.pdf", plot = DNS_Left, width = 6, height = 4)

model2 <- lm(Next_Abs_Expansion ~ Abs_Contraction, data=filter(df_nocovid_Pred,Color=="Before 1980"))
summary(model2)
model2a <- lm(Next_Abs_Expansion ~ Abs_Contraction, data=filter(df_nocovid_Pred,Color=="After 1980"))
summary(model2a)

write.xlsx(df_nocovid_Pred, 'Rep_Figures_Data_Export/Fig10_Export.xlsx')


#Figure 9 (numerous figures)
Dat1949<-read.table("Rep_Data/1949.csv",sep=',',header=T)

Dat1949_long <-Dat1949%>%
  pivot_longer(cols=AK:US, names_to = "state", values_to = "change")

Fig9_1949<-ggplot() + geom_line(data = Dat1949_long, aes(x = Months.from.Trough, y = change, group=state), color="grey", alpha = 0.4, size = 1)+
  theme(legend.position = "none",axis.line = element_line(colour = "black"),panel.background = element_blank(),plot.title = element_text(hjust = 0.5)) +ylim(-1.5,0.1)+
  scale_x_continuous(breaks = seq(-12, 12, 1))+xlab("months from trough date")+ylab("log change in CBUR from trough date")+
  ggtitle("1949")+ geom_line(data = filter(Dat1949_long, state == "US"), aes(x = Months.from.Trough, y = change),size = 2, colour="black")

ggsave("Rep_Figures/Fig9_1949.pdf", plot = Fig9_1949, width = 6, height = 4)

Dat1954<-read.table("Rep_Data/1954.csv",sep=',',header=T)

Dat1954_long <-Dat1954%>%
  pivot_longer(cols=AK:US, names_to = "state", values_to = "change")

Fig9_1954<-ggplot() + geom_line(data = Dat1954_long, aes(x = Months.from.Trough, y = change, group = state), color="grey", alpha = 0.4, size = 1)+
  theme(legend.position = "none",axis.line = element_line(colour = "black"),panel.background = element_blank(),plot.title = element_text(hjust = 0.5)) +ylim(-1.5,0.1)+
  scale_x_continuous(breaks = seq(-12, 12, 1))+xlab("months from trough date")+ylab("log change in CBUR from trough date")+
  ggtitle("1954") + geom_line(data = filter(Dat1954_long, state == "US"), aes(x = Months.from.Trough, y = change),size = 2, colour="black")

ggsave("Rep_Figures/Fig9_1954.pdf", plot = Fig9_1954, width = 6, height = 4)


Dat1958<-read.table("Rep_Data/1958.csv",sep=',',header=T)

Dat1958_long <-Dat1958%>%
  pivot_longer(cols=AK:US, names_to = "state", values_to = "change")

Fig9_1958<-ggplot() + geom_line(data = Dat1958_long, aes(x = Months.from.Trough, y = change, group = state), color="grey", alpha = 0.4, size = 1)+
  theme(legend.position = "none",axis.line = element_line(colour = "black"),panel.background = element_blank(),plot.title = element_text(hjust = 0.5)) +ylim(-1.5,0.1)+
  scale_x_continuous(breaks = seq(-12, 12, 1))+xlab("months from trough date")+ylab("log change in CBUR from trough date")+
  ggtitle("1958") + geom_line(data = filter(Dat1958_long, state == "US"), aes(x = Months.from.Trough, y = change),size = 2, colour="black")

ggsave("Rep_Figures/Fig9_1958.pdf", plot = Fig9_1958, width = 6, height = 4)

Dat1961<-read.table("Rep_Data/1961.csv",sep=',',header=T)

Dat1961_long <-Dat1961%>%
  pivot_longer(cols=AK:US, names_to = "state", values_to = "change")

Fig9_1961<-ggplot() + geom_line(data = Dat1961_long, aes(x = Months.from.Trough, y = change, group = state), color="grey", alpha = 0.4, size = 1)+
  theme(legend.position = "none",axis.line = element_line(colour = "black"),panel.background = element_blank(),plot.title = element_text(hjust = 0.5)) +ylim(-1.5,0.1)+
  scale_x_continuous(breaks = seq(-12, 12, 1))+xlab("months from trough date")+ylab("log change in CBUR from trough date")+
  ggtitle("1961") + geom_line(data = filter(Dat1961_long, state == "US"), aes(x = Months.from.Trough, y = change),size = 2, colour="black")

ggsave("Rep_Figures/Fig9_1961.pdf", plot = Fig9_1961, width = 6, height = 4)

Dat1970<-read.table("Rep_Data/1970.csv",sep=',',header=T)

Dat1970_long <-Dat1970%>%
  pivot_longer(cols=AK:US, names_to = "state", values_to = "change")

Fig9_1970<-ggplot() + geom_line(data = Dat1970_long, aes(x = Months.from.Trough, y = change, group = state), color="grey", alpha = 0.4, size = 1)+
  theme(legend.position = "none",axis.line = element_line(colour = "black"),panel.background = element_blank(),plot.title = element_text(hjust = 0.5)) +ylim(-1.5,0.1)+
  scale_x_continuous(breaks = seq(-12, 12, 1))+xlab("months from trough date")+ylab("log change in CBUR from trough date")+
  ggtitle("1970") + geom_line(data = filter(Dat1970_long, state == "US"), aes(x = Months.from.Trough, y = change),size = 2, colour="black")

ggsave("Rep_Figures/Fig9_1970.pdf", plot = Fig9_1970, width = 6, height = 4)


Dat1975<-read.table("Rep_Data/1975.csv",sep=',',header=T)

Dat1975_long <-Dat1975%>%
  pivot_longer(cols=AK:US, names_to = "state", values_to = "change")

Fig9_1975<-ggplot() + geom_line(data = Dat1975_long, aes(x = Months.from.Trough, y = change, group = state), color="grey", alpha = 0.4, size = 1)+
  theme(legend.position = "none",axis.line = element_line(colour = "black"),panel.background = element_blank(),plot.title = element_text(hjust = 0.5)) +ylim(-1.5,0.1)+
  scale_x_continuous(breaks = seq(-12, 12, 1))+xlab("months from trough date")+ylab("log change in CBUR from trough date")+
  ggtitle("1975") + geom_line(data = filter(Dat1975_long, state == "US"), aes(x = Months.from.Trough, y = change),size = 2, colour="black")

ggsave("Rep_Figures/Fig9_1975.pdf", plot = Fig9_1975, width = 6, height = 4)


Dat1982<-read.table("Rep_Data/1982.csv",sep=',',header=T)

Dat1982_long <-Dat1982%>%
  pivot_longer(cols=AK:US, names_to = "state", values_to = "change")

Fig9_1982<-ggplot() + geom_line(data = Dat1982_long, aes(x = Months.from.Trough, y = change, group = state), color="grey", alpha = 0.4, size = 1)+
  theme(legend.position = "none",axis.line = element_line(colour = "black"),panel.background = element_blank(),plot.title = element_text(hjust = 0.5)) +ylim(-1.5,0.1)+
  scale_x_continuous(breaks = seq(-12, 12, 1))+xlab("months from trough date")+ylab("log change in CBUR from trough date")+
  ggtitle("1982") + geom_line(data = filter(Dat1982_long, state == "US"), aes(x = Months.from.Trough, y = change),size = 2, colour="black")

ggsave("Rep_Figures/Fig9_1982.pdf", plot = Fig9_1982, width = 6, height = 4)


Dat1992<-read.table("Rep_Data/1992.csv",sep=',',header=T)

Dat1992_long <-Dat1992%>%
  pivot_longer(cols=AK:US, names_to = "state", values_to = "change")

Fig9_1992<-ggplot() + geom_line(data = Dat1992_long, aes(x = Months.from.Trough, y = change, group = state), color="grey", alpha = 0.4, size = 1)+
  theme(legend.position = "none",axis.line = element_line(colour = "black"),panel.background = element_blank(),plot.title = element_text(hjust = 0.5)) +ylim(-1.5,0.1)+
  scale_x_continuous(breaks = seq(-12, 12, 1))+xlab("months from trough date")+ylab("log change in CBUR from trough date")+
  ggtitle("1992") + geom_line(data = filter(Dat1992_long, state == "US"), aes(x = Months.from.Trough, y = change),size = 2, colour="black")

ggsave("Rep_Figures/Fig9_1992.pdf", plot = Fig9_1992, width = 6, height = 4)


Dat2003<-read.table("Rep_Data/2003.csv",sep=',',header=T)

Dat2003_long <-Dat2003%>%
  pivot_longer(cols=AK:US, names_to = "state", values_to = "change")

Fig9_2003<-ggplot() + geom_line(data = Dat2003_long, aes(x = Months.from.Trough, y = change, group = state), color="grey", alpha = 0.4, size = 1)+
  theme(legend.position = "none",axis.line = element_line(colour = "black"),panel.background = element_blank(),plot.title = element_text(hjust = 0.5)) +ylim(-1.5,0.1)+
  scale_x_continuous(breaks = seq(-12, 12, 1))+xlab("months from trough date")+ylab("log change in CBUR from trough date")+
  ggtitle("2003") + geom_line(data = filter(Dat2003_long, state == "US"), aes(x = Months.from.Trough, y = change),size = 2, colour="black")

ggsave("Rep_Figures/Fig9_2003.pdf", plot = Fig9_2003, width = 6, height = 4)


Dat2009<-read.table("Rep_Data/2009.csv",sep=',',header=T)

Dat2009_long <-Dat2009%>%
  pivot_longer(cols=AK:US, names_to = "state", values_to = "change")

Fig9_2009<-ggplot() + geom_line(data = Dat2009_long, aes(x = Months.from.Trough, y = change, group = state), color="grey", alpha = 0.4, size = 1)+
  theme(legend.position = "none",axis.line = element_line(colour = "black"),panel.background = element_blank(),plot.title = element_text(hjust = 0.5)) +ylim(-1.5,0.1)+
  scale_x_continuous(breaks = seq(-12, 12, 1))+xlab("months from trough date")+ylab("log change in CBUR from trough date")+
  ggtitle("2009") + geom_line(data = filter(Dat2009_long, state == "US"), aes(x = Months.from.Trough, y = change),size = 2, colour="black")

ggsave("Rep_Figures/Fig9_2009.pdf", plot = Fig9_2009, width = 6, height = 4)


Dat2020<-read.table("Rep_Data/2020.csv",sep=',',header=T)

Dat2020_long <-Dat2020%>%
  pivot_longer(cols=AK:US, names_to = "state", values_to = "change")

Fig9_2020<-ggplot() + geom_line(data = Dat2020_long, aes(x = Months.from.Trough, y = change, group = state), color="grey", alpha = 0.4, size = 1)+
  theme(legend.position = "none",axis.line = element_line(colour = "black"),panel.background = element_blank(),plot.title = element_text(hjust = 0.5)) +ylim(-1.5,0.1)+
  scale_x_continuous(breaks = seq(-12, 12, 1))+xlab("months from trough date")+ylab("log change in CBUR from trough date")+
  ggtitle("2020") + geom_line(data = filter(Dat2020_long, state == "US"), aes(x = Months.from.Trough, y = change),size = 2, colour="black")

ggsave("Rep_Figures/Fig9_2020.pdf", plot = Fig9_2020, width = 6, height = 4)

names <- list('1949' = Dat1949_long, '1954' = Dat1954_long, '1958' = Dat1958_long, '1961' = Dat1961_long, '1970' = Dat1970_long, '1975' = Dat1975_long, '1982' = Dat1982_long, '1992' = Dat1992_long, '2003' = Dat2003_long, '2009' = Dat2009_long, '2020' = Dat2020_long)
write.xlsx(names, file = 'Rep_Figures_Data_Export/Fig9_Export.xlsx')


#END OF FILE